document.addEventListener('DOMContentLoaded', function() {
    const sendButton = document.getElementById('sendButton');
    const loadingStatus = document.getElementById('loadingStatus');
    const successStatus = document.getElementById('successStatus');
    const errorStatus = document.getElementById('errorStatus');
    const sellerIdInput = document.getElementById('sellerId');

    function showStatus(element, message = '') {
        loadingStatus.style.display = 'none';
        successStatus.style.display = 'none';
        errorStatus.style.display = 'none';
        
        if (message && element.querySelector('.status-text')) {
            element.querySelector('.status-text').textContent = message;
        }
        
        element.style.display = 'block';
    }

    // Разрешаем вводить только цифры
    sellerIdInput.addEventListener('input', function() {
        this.value = this.value.replace(/[^0-9]/g, '');
    });
    
    // Предотвращаем вставку не-цифр
    sellerIdInput.addEventListener('paste', function(e) {
        const pasteData = e.clipboardData.getData('text');
        if (!/^\d+$/.test(pasteData)) {
            e.preventDefault();
        }
    });

    // Функция для сохранения текста в хранилище Chrome
    function saveText() {
        const text = sellerIdInput.value;
        
        chrome.storage.local.set({ savedText: text }, function() {
            // Тонкое уведомление сохранения
            console.log('✓ Seller ID сохранен');
        });
    }
    
    // Функция для загрузки текста из хранилища
    function loadText() {
        chrome.storage.local.get(['savedText'], function(result) {
            if (result.savedText) {
                sellerIdInput.value = result.savedText.replace(/[^0-9]/g, '');
            }
        });
    }

    // Функция для сохранения текста при каждом изменении (автосохранение)
    function setupAutoSave() {
        let saveTimeout;
        
        sellerIdInput.addEventListener('input', function() {
            // Очищаем предыдущий таймер
            if (saveTimeout) {
                clearTimeout(saveTimeout);
            }
            
            // Устанавливаем новый таймер на 1 секунду
            saveTimeout = setTimeout(() => {
                saveText();
            }, 1000);
        });
    }

    // Инициализация текстового поля
    function initTextField() {
        // Загружаем сохраненный текст при загрузке
        loadText();
        
        // Настраиваем автосохранение
        setupAutoSave();
    }

    // Вызываем инициализацию текстового поля
    initTextField();

    sendButton.addEventListener('click', async function() {
        sendButton.disabled = true;
        showStatus(loadingStatus);
        
        try {
            console.log('🎯 Начало отправки данных...');
            
            // Получаем активную вкладку
            const [tab] = await chrome.tabs.query({ active: true, currentWindow: true });
            console.log('🌐 Текущая вкладка:', tab.url);
            console.log('📝 Заголовок страницы:', tab.title);
            
            // Получаем сохраненный Seller ID
            const sellerId = sellerIdInput.value || '';
            console.log('📝 Seller ID:', sellerId);

            // Проверяем, заполнен ли Seller ID
            if (!sellerId.trim()) {
                throw new Error('Не заполнен Seller ID');
            }

            // Выполняем скрипт на странице для получения HTML
            const results = await chrome.scripting.executeScript({
                target: { tabId: tab.id },
                function: getPageHTML
            });

            const pageHTML = results[0].result;
            console.log('📄 HTML получен, длина:', pageHTML.length, 'символов');

            // Подготавливаем данные для отправки
            const requestData = {
                html: JSON.stringify(pageHTML),
                url: tab.url,           // URL страницы
                title: tab.title,       // Заголовок страницы
                timestamp: new Date().toISOString(),
                userAgent: navigator.userAgent,
                extensionVersion: '1.0',
                sellerId: sellerId      // Добавляем Seller ID
            };

            console.log('🚀 Отправка запроса на:', 'https://srv7.mp-raketa.ru/DatabaseManager1C_new_b24/hs/ozon_ext/in');
            console.log('📦 Отправляемые данные:', {
                url: requestData.url,
                title: requestData.title,
                html_length: requestData.html.length,
                sellerId: sellerId,
                timestamp: requestData.timestamp
            });

            // Отправляем данные на сервер
            const response = await fetch('https://srv7.mp-raketa.ru/DatabaseManager1C_new_b24/hs/ozon_ext/in', {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/json',
                    'Accept': 'application/json',
                },
                body: JSON.stringify(requestData),
                signal: AbortSignal.timeout(30000)
            });

            console.log('📡 Статус ответа:', response.status);
            console.log('✅ OK:', response.ok);

            if (response.ok) {
                try {
                    const responseText = await response.text();
                    console.log('📨 Ответ сервера:', responseText);
                    showStatus(successStatus, `Данные успешно отправлены!`);
                } catch (e) {
                    console.log('📨 Ответ сервера (без текста)');
                    showStatus(successStatus, `Данные успешно отправлены!`);
                }
            } else {
                const errorText = await response.text();
                console.error('❌ Ошибка сервера:', response.status, errorText);
                throw new Error(`HTTP error! status: ${response.status}, response: ${errorText}`);
            }

        } catch (error) {
            console.error('💥 Полная ошибка:', error);
            
            let errorMessage = 'Ошибка при отправке данных';
            
            if (error.message === 'Не заполнен Seller ID') {
                errorMessage = 'Пожалуйста, введите Seller ID (только цифры)';
            } else if (error.name === 'AbortError') {
                errorMessage = 'Таймаут запроса (30 секунд)';
            } else if (error.message.includes('Failed to fetch')) {
                errorMessage = 'Не удалось подключиться к серверу. Проверьте:\n- Доступность сервера\n- CORS настройки\n- Сетевое соединение';
            } else if (error.message.includes('HTTP error')) {
                errorMessage = `Ошибка сервера: ${error.message}`;
            }
            
            // Добавляем информацию о URL в сообщение об ошибке
            try {
                const [tab] = await chrome.tabs.query({ active: true, currentWindow: true });
                errorMessage += `\nURL: ${tab.url}`;
            } catch (tabError) {
                console.log('Не удалось получить URL для ошибки');
            }
            
            showStatus(errorStatus, errorMessage);
        } finally {
            sendButton.disabled = false;
            
            // Автоматически закрываем popup через 3 секунды после успешной отправки
            setTimeout(() => {
                if (!sendButton.disabled && successStatus.style.display === 'block') {
                    window.close();
                }
            }, 3000);
        }
    });
});

// Функция, которая выполняется на странице для получения HTML
function getPageHTML() {
    return document.documentElement.outerHTML;
}